'use strict';

/* --------------------------------------------------------------
 add_tracking_number.js 2016-09-12
 Gambio GmbH
 http://www.gambio.de
 Copyright (c) 2021 Gambio GmbH
 Released under the GNU General Public License (Version 2)
 [http://www.gnu.org/licenses/gpl-2.0.html]
 --------------------------------------------------------------
 */

/**
 * Add Tracking Number Modal Controller
 *
 * Handles the functionality of the "Add Tracking Number" modal.
 */
gx.controllers.module('add_tracking_number', ['modal', gx.source + '/libs/info_box'], function (data) {

    'use strict';

    // ------------------------------------------------------------------------
    // VARIABLES
    // ------------------------------------------------------------------------

    /**
     * Module Selector
     *
     * @type {jQuery}
     */

    var $this = $(this);

    /**
     * Module Instance
     *
     * @type {Object}
     */
    var module = {};

    // ------------------------------------------------------------------------
    // FUNCTIONS
    // ------------------------------------------------------------------------

    /**
     * Stores the tracking number for a specific order.
     *
     * @param {jQuery.Event} event
     */
    function _onStoreTrackingNumberClick(event) {
        event.preventDefault();

        var orderId = $this.data('orderId');
        var parcelServiceId = $('#delivery-service').find('option:selected').val();
        var trackingNumber = $('input[name="tracking-number"]').val();
        var isReturnDelivery = $('input[name="isReturnDelivery"]').prop('checked');
        var shipmentType = $('input[name="shipmentType"]').val();

        // Make an AJAX call to store the tracking number if one was provided.
        if (trackingNumber.length) {
            $.ajax({
                url: './admin.php?do=OrdersModalsAjax/StoreTrackingNumber',
                data: {
                    orderId: orderId,
                    trackingNumber: trackingNumber,
                    parcelServiceId: parcelServiceId,
                    isReturnDelivery: isReturnDelivery,
                    shipmentType: shipmentType,
                    pageToken: jse.core.config.get('pageToken')
                },
                method: 'POST',
                dataType: 'JSON'
            }).done(function (response) {
                $this.modal('hide');
                jse.libs.info_box.addSuccessMessage(jse.core.lang.translate('ADD_TRACKING_NUMBER_SUCCESS', 'admin_orders'));
                $('.table-main').DataTable().ajax.reload(null, false);
            }).fail(function (jqXHR, textStatus, errorThrown) {
                jse.libs.modal.message({
                    title: jse.core.lang.translate('error', 'messages'),
                    content: jse.core.lang.translate('ADD_TRACKING_NUMBER_ERROR', 'admin_orders')
                });
                jse.core.debug.error('Store Tracking Number Error', jqXHR, textStatus, errorThrown);
            });
        } else {
            // Show an error message
            var $modalFooter = $this.find('.modal-footer');
            var errorMessage = jse.core.lang.translate('TXT_SAVE_ERROR', 'admin_general');

            // Remove error message
            $modalFooter.find('span').remove();
            $modalFooter.prepend('<span class="text-danger">' + errorMessage + '</span>');
        }
    }

    /**
     * On Add Tracking Number Modal Hidden
     *
     * Reset the tracking number modal.
     */
    function _onAddTrackingNumberModalHidden() {
        $(this).find('#tracking-number').val('');
        $('input[name="isReturnDelivery"]').prop('checked', false).trigger('change');
        $('input[name="shipmentType"]').val('');
        $(this).find('.modal-footer span').remove();
    }

    /**
     * On Add Tracking Number Modal Show
     *
     * Handles the event for storing a a tracking number from the tracking number modal.
     *
     * @param {jQuery.Event} event
     */
    function _onAddTrackingNumberModalShow(event) {
        event.stopPropagation();
        // Element which invoked the tracking number modal.
        $(this).data('orderId', $(event.relatedTarget).data('orderId'));
    }

    /**
     * Checks if the enter key was pressed and delegates to
     * the tracking number store method.
     *
     * @param {jQuery.Event} event
     */
    function _saveOnPressedEnterKey(event) {
        var keyCode = event.keyCode ? event.keyCode : event.which;

        if (keyCode === 13) {
            _onStoreTrackingNumberClick(event);
        }
    }

    // ------------------------------------------------------------------------
    // INITIALIZATION
    // ------------------------------------------------------------------------

    module.init = function (done) {
        $this.on('show.bs.modal', _onAddTrackingNumberModalShow).on('hidden.bs.modal', _onAddTrackingNumberModalHidden).on('click', '#store-tracking-number', _onStoreTrackingNumberClick).on('keypress', _saveOnPressedEnterKey);

        done();
    };

    return module;
});
//# sourceMappingURL=data:application/json;charset=utf8;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbIm9yZGVycy9tb2RhbHMvYWRkX3RyYWNraW5nX251bWJlci5qcyJdLCJuYW1lcyI6WyJneCIsImNvbnRyb2xsZXJzIiwibW9kdWxlIiwic291cmNlIiwiZGF0YSIsIiR0aGlzIiwiJCIsIl9vblN0b3JlVHJhY2tpbmdOdW1iZXJDbGljayIsImV2ZW50IiwicHJldmVudERlZmF1bHQiLCJvcmRlcklkIiwicGFyY2VsU2VydmljZUlkIiwiZmluZCIsInZhbCIsInRyYWNraW5nTnVtYmVyIiwiaXNSZXR1cm5EZWxpdmVyeSIsInByb3AiLCJzaGlwbWVudFR5cGUiLCJsZW5ndGgiLCJhamF4IiwidXJsIiwicGFnZVRva2VuIiwianNlIiwiY29yZSIsImNvbmZpZyIsImdldCIsIm1ldGhvZCIsImRhdGFUeXBlIiwiZG9uZSIsInJlc3BvbnNlIiwibW9kYWwiLCJsaWJzIiwiaW5mb19ib3giLCJhZGRTdWNjZXNzTWVzc2FnZSIsImxhbmciLCJ0cmFuc2xhdGUiLCJEYXRhVGFibGUiLCJyZWxvYWQiLCJmYWlsIiwianFYSFIiLCJ0ZXh0U3RhdHVzIiwiZXJyb3JUaHJvd24iLCJtZXNzYWdlIiwidGl0bGUiLCJjb250ZW50IiwiZGVidWciLCJlcnJvciIsIiRtb2RhbEZvb3RlciIsImVycm9yTWVzc2FnZSIsInJlbW92ZSIsInByZXBlbmQiLCJfb25BZGRUcmFja2luZ051bWJlck1vZGFsSGlkZGVuIiwidHJpZ2dlciIsIl9vbkFkZFRyYWNraW5nTnVtYmVyTW9kYWxTaG93Iiwic3RvcFByb3BhZ2F0aW9uIiwicmVsYXRlZFRhcmdldCIsIl9zYXZlT25QcmVzc2VkRW50ZXJLZXkiLCJrZXlDb2RlIiwid2hpY2giLCJpbml0Iiwib24iXSwibWFwcGluZ3MiOiI7O0FBQUE7Ozs7Ozs7Ozs7QUFVQTs7Ozs7QUFLQUEsR0FBR0MsV0FBSCxDQUFlQyxNQUFmLENBQXNCLHFCQUF0QixFQUE2QyxDQUFDLE9BQUQsRUFBYUYsR0FBR0csTUFBaEIsb0JBQTdDLEVBQXNGLFVBQVNDLElBQVQsRUFBZTs7QUFFakc7O0FBRUE7QUFDQTtBQUNBOztBQUVBOzs7Ozs7QUFLQSxRQUFNQyxRQUFRQyxFQUFFLElBQUYsQ0FBZDs7QUFFQTs7Ozs7QUFLQSxRQUFNSixTQUFTLEVBQWY7O0FBRUE7QUFDQTtBQUNBOztBQUVBOzs7OztBQUtBLGFBQVNLLDJCQUFULENBQXFDQyxLQUFyQyxFQUE0QztBQUN4Q0EsY0FBTUMsY0FBTjs7QUFFQSxZQUFNQyxVQUFVTCxNQUFNRCxJQUFOLENBQVcsU0FBWCxDQUFoQjtBQUNBLFlBQU1PLGtCQUFrQkwsRUFBRSxtQkFBRixFQUF1Qk0sSUFBdkIsQ0FBNEIsaUJBQTVCLEVBQStDQyxHQUEvQyxFQUF4QjtBQUNBLFlBQU1DLGlCQUFpQlIsRUFBRSwrQkFBRixFQUFtQ08sR0FBbkMsRUFBdkI7QUFDQSxZQUFNRSxtQkFBbUJULEVBQUUsZ0NBQUYsRUFBb0NVLElBQXBDLENBQXlDLFNBQXpDLENBQXpCO0FBQ0EsWUFBTUMsZUFBZVgsRUFBRSw0QkFBRixFQUFnQ08sR0FBaEMsRUFBckI7O0FBRUE7QUFDQSxZQUFJQyxlQUFlSSxNQUFuQixFQUEyQjtBQUN2QlosY0FBRWEsSUFBRixDQUFPO0FBQ0hDLHFCQUFLLHFEQURGO0FBRUhoQixzQkFBTTtBQUNGTSxvQ0FERTtBQUVGSSxrREFGRTtBQUdGSCxvREFIRTtBQUlGSSxzREFKRTtBQUtGRSw4Q0FMRTtBQU1GSSwrQkFBV0MsSUFBSUMsSUFBSixDQUFTQyxNQUFULENBQWdCQyxHQUFoQixDQUFvQixXQUFwQjtBQU5ULGlCQUZIO0FBVUhDLHdCQUFRLE1BVkw7QUFXSEMsMEJBQVU7QUFYUCxhQUFQLEVBYUtDLElBYkwsQ0FhVSxVQUFTQyxRQUFULEVBQW1CO0FBQ3JCeEIsc0JBQU15QixLQUFOLENBQVksTUFBWjtBQUNBUixvQkFBSVMsSUFBSixDQUFTQyxRQUFULENBQWtCQyxpQkFBbEIsQ0FDSVgsSUFBSUMsSUFBSixDQUFTVyxJQUFULENBQWNDLFNBQWQsQ0FBd0IsNkJBQXhCLEVBQXVELGNBQXZELENBREo7QUFFQTdCLGtCQUFFLGFBQUYsRUFBaUI4QixTQUFqQixHQUE2QmpCLElBQTdCLENBQWtDa0IsTUFBbEMsQ0FBeUMsSUFBekMsRUFBK0MsS0FBL0M7QUFDSCxhQWxCTCxFQW1CS0MsSUFuQkwsQ0FtQlUsVUFBU0MsS0FBVCxFQUFnQkMsVUFBaEIsRUFBNEJDLFdBQTVCLEVBQXlDO0FBQzNDbkIsb0JBQUlTLElBQUosQ0FBU0QsS0FBVCxDQUFlWSxPQUFmLENBQXVCO0FBQ25CQywyQkFBT3JCLElBQUlDLElBQUosQ0FBU1csSUFBVCxDQUFjQyxTQUFkLENBQXdCLE9BQXhCLEVBQWlDLFVBQWpDLENBRFk7QUFFbkJTLDZCQUFTdEIsSUFBSUMsSUFBSixDQUFTVyxJQUFULENBQWNDLFNBQWQsQ0FBd0IsMkJBQXhCLEVBQXFELGNBQXJEO0FBRlUsaUJBQXZCO0FBSUFiLG9CQUFJQyxJQUFKLENBQVNzQixLQUFULENBQWVDLEtBQWYsQ0FBcUIsNkJBQXJCLEVBQW9EUCxLQUFwRCxFQUEyREMsVUFBM0QsRUFBdUVDLFdBQXZFO0FBQ0gsYUF6Qkw7QUEwQkgsU0EzQkQsTUEyQk87QUFDSDtBQUNBLGdCQUFNTSxlQUFlMUMsTUFBTU8sSUFBTixDQUFXLGVBQVgsQ0FBckI7QUFDQSxnQkFBTW9DLGVBQWUxQixJQUFJQyxJQUFKLENBQVNXLElBQVQsQ0FBY0MsU0FBZCxDQUF3QixnQkFBeEIsRUFBMEMsZUFBMUMsQ0FBckI7O0FBRUE7QUFDQVkseUJBQWFuQyxJQUFiLENBQWtCLE1BQWxCLEVBQTBCcUMsTUFBMUI7QUFDQUYseUJBQWFHLE9BQWIsZ0NBQWtERixZQUFsRDtBQUNIO0FBQ0o7O0FBRUQ7Ozs7O0FBS0EsYUFBU0csK0JBQVQsR0FBMkM7QUFDdkM3QyxVQUFFLElBQUYsRUFBUU0sSUFBUixDQUFhLGtCQUFiLEVBQWlDQyxHQUFqQyxDQUFxQyxFQUFyQztBQUNBUCxVQUFFLGdDQUFGLEVBQW9DVSxJQUFwQyxDQUF5QyxTQUF6QyxFQUFvRCxLQUFwRCxFQUEyRG9DLE9BQTNELENBQW1FLFFBQW5FO0FBQ0E5QyxVQUFFLDRCQUFGLEVBQWdDTyxHQUFoQyxDQUFvQyxFQUFwQztBQUNBUCxVQUFFLElBQUYsRUFBUU0sSUFBUixDQUFhLG9CQUFiLEVBQW1DcUMsTUFBbkM7QUFDSDs7QUFHRDs7Ozs7OztBQU9BLGFBQVNJLDZCQUFULENBQXVDN0MsS0FBdkMsRUFBOEM7QUFDMUNBLGNBQU04QyxlQUFOO0FBQ0E7QUFDQWhELFVBQUUsSUFBRixFQUFRRixJQUFSLENBQWEsU0FBYixFQUF3QkUsRUFBRUUsTUFBTStDLGFBQVIsRUFBdUJuRCxJQUF2QixDQUE0QixTQUE1QixDQUF4QjtBQUNIOztBQUVEOzs7Ozs7QUFNQSxhQUFTb0Qsc0JBQVQsQ0FBZ0NoRCxLQUFoQyxFQUF1QztBQUNuQyxZQUFNaUQsVUFBVWpELE1BQU1pRCxPQUFOLEdBQWdCakQsTUFBTWlELE9BQXRCLEdBQWdDakQsTUFBTWtELEtBQXREOztBQUVBLFlBQUlELFlBQVksRUFBaEIsRUFBb0I7QUFDaEJsRCx3Q0FBNEJDLEtBQTVCO0FBQ0g7QUFDSjs7QUFFRDtBQUNBO0FBQ0E7O0FBRUFOLFdBQU95RCxJQUFQLEdBQWMsVUFBUy9CLElBQVQsRUFBZTtBQUN6QnZCLGNBQ0t1RCxFQURMLENBQ1EsZUFEUixFQUN5QlAsNkJBRHpCLEVBRUtPLEVBRkwsQ0FFUSxpQkFGUixFQUUyQlQsK0JBRjNCLEVBR0tTLEVBSEwsQ0FHUSxPQUhSLEVBR2lCLHdCQUhqQixFQUcyQ3JELDJCQUgzQyxFQUlLcUQsRUFKTCxDQUlRLFVBSlIsRUFJb0JKLHNCQUpwQjs7QUFNQTVCO0FBQ0gsS0FSRDs7QUFVQSxXQUFPMUIsTUFBUDtBQUNILENBdElEIiwiZmlsZSI6Im9yZGVycy9tb2RhbHMvYWRkX3RyYWNraW5nX251bWJlci5qcyIsInNvdXJjZXNDb250ZW50IjpbIi8qIC0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tXG4gYWRkX3RyYWNraW5nX251bWJlci5qcyAyMDE2LTA5LTEyXG4gR2FtYmlvIEdtYkhcbiBodHRwOi8vd3d3LmdhbWJpby5kZVxuIENvcHlyaWdodCAoYykgMjAyMSBHYW1iaW8gR21iSFxuIFJlbGVhc2VkIHVuZGVyIHRoZSBHTlUgR2VuZXJhbCBQdWJsaWMgTGljZW5zZSAoVmVyc2lvbiAyKVxuIFtodHRwOi8vd3d3LmdudS5vcmcvbGljZW5zZXMvZ3BsLTIuMC5odG1sXVxuIC0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tXG4gKi9cblxuLyoqXG4gKiBBZGQgVHJhY2tpbmcgTnVtYmVyIE1vZGFsIENvbnRyb2xsZXJcbiAqXG4gKiBIYW5kbGVzIHRoZSBmdW5jdGlvbmFsaXR5IG9mIHRoZSBcIkFkZCBUcmFja2luZyBOdW1iZXJcIiBtb2RhbC5cbiAqL1xuZ3guY29udHJvbGxlcnMubW9kdWxlKCdhZGRfdHJhY2tpbmdfbnVtYmVyJywgWydtb2RhbCcsIGAke2d4LnNvdXJjZX0vbGlicy9pbmZvX2JveGBdLCBmdW5jdGlvbihkYXRhKSB7XG4gICAgXG4gICAgJ3VzZSBzdHJpY3QnO1xuICAgIFxuICAgIC8vIC0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLVxuICAgIC8vIFZBUklBQkxFU1xuICAgIC8vIC0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLVxuICAgIFxuICAgIC8qKlxuICAgICAqIE1vZHVsZSBTZWxlY3RvclxuICAgICAqXG4gICAgICogQHR5cGUge2pRdWVyeX1cbiAgICAgKi9cbiAgICBjb25zdCAkdGhpcyA9ICQodGhpcyk7XG4gICAgXG4gICAgLyoqXG4gICAgICogTW9kdWxlIEluc3RhbmNlXG4gICAgICpcbiAgICAgKiBAdHlwZSB7T2JqZWN0fVxuICAgICAqL1xuICAgIGNvbnN0IG1vZHVsZSA9IHt9O1xuICAgIFxuICAgIC8vIC0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLVxuICAgIC8vIEZVTkNUSU9OU1xuICAgIC8vIC0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLVxuICAgIFxuICAgIC8qKlxuICAgICAqIFN0b3JlcyB0aGUgdHJhY2tpbmcgbnVtYmVyIGZvciBhIHNwZWNpZmljIG9yZGVyLlxuICAgICAqXG4gICAgICogQHBhcmFtIHtqUXVlcnkuRXZlbnR9IGV2ZW50XG4gICAgICovXG4gICAgZnVuY3Rpb24gX29uU3RvcmVUcmFja2luZ051bWJlckNsaWNrKGV2ZW50KSB7XG4gICAgICAgIGV2ZW50LnByZXZlbnREZWZhdWx0KCk7XG4gICAgICAgIFxuICAgICAgICBjb25zdCBvcmRlcklkID0gJHRoaXMuZGF0YSgnb3JkZXJJZCcpO1xuICAgICAgICBjb25zdCBwYXJjZWxTZXJ2aWNlSWQgPSAkKCcjZGVsaXZlcnktc2VydmljZScpLmZpbmQoJ29wdGlvbjpzZWxlY3RlZCcpLnZhbCgpO1xuICAgICAgICBjb25zdCB0cmFja2luZ051bWJlciA9ICQoJ2lucHV0W25hbWU9XCJ0cmFja2luZy1udW1iZXJcIl0nKS52YWwoKTtcbiAgICAgICAgY29uc3QgaXNSZXR1cm5EZWxpdmVyeSA9ICQoJ2lucHV0W25hbWU9XCJpc1JldHVybkRlbGl2ZXJ5XCJdJykucHJvcCgnY2hlY2tlZCcpO1xuICAgICAgICBjb25zdCBzaGlwbWVudFR5cGUgPSAkKCdpbnB1dFtuYW1lPVwic2hpcG1lbnRUeXBlXCJdJykudmFsKCk7XG4gICAgICAgIFxuICAgICAgICAvLyBNYWtlIGFuIEFKQVggY2FsbCB0byBzdG9yZSB0aGUgdHJhY2tpbmcgbnVtYmVyIGlmIG9uZSB3YXMgcHJvdmlkZWQuXG4gICAgICAgIGlmICh0cmFja2luZ051bWJlci5sZW5ndGgpIHtcbiAgICAgICAgICAgICQuYWpheCh7XG4gICAgICAgICAgICAgICAgdXJsOiAnLi9hZG1pbi5waHA/ZG89T3JkZXJzTW9kYWxzQWpheC9TdG9yZVRyYWNraW5nTnVtYmVyJyxcbiAgICAgICAgICAgICAgICBkYXRhOiB7XG4gICAgICAgICAgICAgICAgICAgIG9yZGVySWQsXG4gICAgICAgICAgICAgICAgICAgIHRyYWNraW5nTnVtYmVyLFxuICAgICAgICAgICAgICAgICAgICBwYXJjZWxTZXJ2aWNlSWQsXG4gICAgICAgICAgICAgICAgICAgIGlzUmV0dXJuRGVsaXZlcnksXG4gICAgICAgICAgICAgICAgICAgIHNoaXBtZW50VHlwZSxcbiAgICAgICAgICAgICAgICAgICAgcGFnZVRva2VuOiBqc2UuY29yZS5jb25maWcuZ2V0KCdwYWdlVG9rZW4nKVxuICAgICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICAgICAgbWV0aG9kOiAnUE9TVCcsXG4gICAgICAgICAgICAgICAgZGF0YVR5cGU6ICdKU09OJ1xuICAgICAgICAgICAgfSlcbiAgICAgICAgICAgICAgICAuZG9uZShmdW5jdGlvbihyZXNwb25zZSkge1xuICAgICAgICAgICAgICAgICAgICAkdGhpcy5tb2RhbCgnaGlkZScpO1xuICAgICAgICAgICAgICAgICAgICBqc2UubGlicy5pbmZvX2JveC5hZGRTdWNjZXNzTWVzc2FnZShcbiAgICAgICAgICAgICAgICAgICAgICAgIGpzZS5jb3JlLmxhbmcudHJhbnNsYXRlKCdBRERfVFJBQ0tJTkdfTlVNQkVSX1NVQ0NFU1MnLCAnYWRtaW5fb3JkZXJzJykpO1xuICAgICAgICAgICAgICAgICAgICAkKCcudGFibGUtbWFpbicpLkRhdGFUYWJsZSgpLmFqYXgucmVsb2FkKG51bGwsIGZhbHNlKTtcbiAgICAgICAgICAgICAgICB9KVxuICAgICAgICAgICAgICAgIC5mYWlsKGZ1bmN0aW9uKGpxWEhSLCB0ZXh0U3RhdHVzLCBlcnJvclRocm93bikge1xuICAgICAgICAgICAgICAgICAgICBqc2UubGlicy5tb2RhbC5tZXNzYWdlKHtcbiAgICAgICAgICAgICAgICAgICAgICAgIHRpdGxlOiBqc2UuY29yZS5sYW5nLnRyYW5zbGF0ZSgnZXJyb3InLCAnbWVzc2FnZXMnKSxcbiAgICAgICAgICAgICAgICAgICAgICAgIGNvbnRlbnQ6IGpzZS5jb3JlLmxhbmcudHJhbnNsYXRlKCdBRERfVFJBQ0tJTkdfTlVNQkVSX0VSUk9SJywgJ2FkbWluX29yZGVycycpXG4gICAgICAgICAgICAgICAgICAgIH0pO1xuICAgICAgICAgICAgICAgICAgICBqc2UuY29yZS5kZWJ1Zy5lcnJvcignU3RvcmUgVHJhY2tpbmcgTnVtYmVyIEVycm9yJywganFYSFIsIHRleHRTdGF0dXMsIGVycm9yVGhyb3duKTtcbiAgICAgICAgICAgICAgICB9KTtcbiAgICAgICAgfSBlbHNlIHtcbiAgICAgICAgICAgIC8vIFNob3cgYW4gZXJyb3IgbWVzc2FnZVxuICAgICAgICAgICAgY29uc3QgJG1vZGFsRm9vdGVyID0gJHRoaXMuZmluZCgnLm1vZGFsLWZvb3RlcicpO1xuICAgICAgICAgICAgY29uc3QgZXJyb3JNZXNzYWdlID0ganNlLmNvcmUubGFuZy50cmFuc2xhdGUoJ1RYVF9TQVZFX0VSUk9SJywgJ2FkbWluX2dlbmVyYWwnKTtcbiAgICAgICAgICAgIFxuICAgICAgICAgICAgLy8gUmVtb3ZlIGVycm9yIG1lc3NhZ2VcbiAgICAgICAgICAgICRtb2RhbEZvb3Rlci5maW5kKCdzcGFuJykucmVtb3ZlKCk7XG4gICAgICAgICAgICAkbW9kYWxGb290ZXIucHJlcGVuZChgPHNwYW4gY2xhc3M9XCJ0ZXh0LWRhbmdlclwiPiR7ZXJyb3JNZXNzYWdlfTwvc3Bhbj5gKTtcbiAgICAgICAgfVxuICAgIH1cbiAgICBcbiAgICAvKipcbiAgICAgKiBPbiBBZGQgVHJhY2tpbmcgTnVtYmVyIE1vZGFsIEhpZGRlblxuICAgICAqXG4gICAgICogUmVzZXQgdGhlIHRyYWNraW5nIG51bWJlciBtb2RhbC5cbiAgICAgKi9cbiAgICBmdW5jdGlvbiBfb25BZGRUcmFja2luZ051bWJlck1vZGFsSGlkZGVuKCkge1xuICAgICAgICAkKHRoaXMpLmZpbmQoJyN0cmFja2luZy1udW1iZXInKS52YWwoJycpO1xuICAgICAgICAkKCdpbnB1dFtuYW1lPVwiaXNSZXR1cm5EZWxpdmVyeVwiXScpLnByb3AoJ2NoZWNrZWQnLCBmYWxzZSkudHJpZ2dlcignY2hhbmdlJyk7XG4gICAgICAgICQoJ2lucHV0W25hbWU9XCJzaGlwbWVudFR5cGVcIl0nKS52YWwoJycpO1xuICAgICAgICAkKHRoaXMpLmZpbmQoJy5tb2RhbC1mb290ZXIgc3BhbicpLnJlbW92ZSgpO1xuICAgIH1cbiAgICBcbiAgICBcbiAgICAvKipcbiAgICAgKiBPbiBBZGQgVHJhY2tpbmcgTnVtYmVyIE1vZGFsIFNob3dcbiAgICAgKlxuICAgICAqIEhhbmRsZXMgdGhlIGV2ZW50IGZvciBzdG9yaW5nIGEgYSB0cmFja2luZyBudW1iZXIgZnJvbSB0aGUgdHJhY2tpbmcgbnVtYmVyIG1vZGFsLlxuICAgICAqXG4gICAgICogQHBhcmFtIHtqUXVlcnkuRXZlbnR9IGV2ZW50XG4gICAgICovXG4gICAgZnVuY3Rpb24gX29uQWRkVHJhY2tpbmdOdW1iZXJNb2RhbFNob3coZXZlbnQpIHtcbiAgICAgICAgZXZlbnQuc3RvcFByb3BhZ2F0aW9uKCk7XG4gICAgICAgIC8vIEVsZW1lbnQgd2hpY2ggaW52b2tlZCB0aGUgdHJhY2tpbmcgbnVtYmVyIG1vZGFsLlxuICAgICAgICAkKHRoaXMpLmRhdGEoJ29yZGVySWQnLCAkKGV2ZW50LnJlbGF0ZWRUYXJnZXQpLmRhdGEoJ29yZGVySWQnKSk7XG4gICAgfVxuICAgIFxuICAgIC8qKlxuICAgICAqIENoZWNrcyBpZiB0aGUgZW50ZXIga2V5IHdhcyBwcmVzc2VkIGFuZCBkZWxlZ2F0ZXMgdG9cbiAgICAgKiB0aGUgdHJhY2tpbmcgbnVtYmVyIHN0b3JlIG1ldGhvZC5cbiAgICAgKlxuICAgICAqIEBwYXJhbSB7alF1ZXJ5LkV2ZW50fSBldmVudFxuICAgICAqL1xuICAgIGZ1bmN0aW9uIF9zYXZlT25QcmVzc2VkRW50ZXJLZXkoZXZlbnQpIHtcbiAgICAgICAgY29uc3Qga2V5Q29kZSA9IGV2ZW50LmtleUNvZGUgPyBldmVudC5rZXlDb2RlIDogZXZlbnQud2hpY2g7XG4gICAgICAgIFxuICAgICAgICBpZiAoa2V5Q29kZSA9PT0gMTMpIHtcbiAgICAgICAgICAgIF9vblN0b3JlVHJhY2tpbmdOdW1iZXJDbGljayhldmVudCk7XG4gICAgICAgIH1cbiAgICB9XG4gICAgXG4gICAgLy8gLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tXG4gICAgLy8gSU5JVElBTElaQVRJT05cbiAgICAvLyAtLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS1cbiAgICBcbiAgICBtb2R1bGUuaW5pdCA9IGZ1bmN0aW9uKGRvbmUpIHtcbiAgICAgICAgJHRoaXNcbiAgICAgICAgICAgIC5vbignc2hvdy5icy5tb2RhbCcsIF9vbkFkZFRyYWNraW5nTnVtYmVyTW9kYWxTaG93KVxuICAgICAgICAgICAgLm9uKCdoaWRkZW4uYnMubW9kYWwnLCBfb25BZGRUcmFja2luZ051bWJlck1vZGFsSGlkZGVuKVxuICAgICAgICAgICAgLm9uKCdjbGljaycsICcjc3RvcmUtdHJhY2tpbmctbnVtYmVyJywgX29uU3RvcmVUcmFja2luZ051bWJlckNsaWNrKVxuICAgICAgICAgICAgLm9uKCdrZXlwcmVzcycsIF9zYXZlT25QcmVzc2VkRW50ZXJLZXkpO1xuICAgICAgICBcbiAgICAgICAgZG9uZSgpO1xuICAgIH07XG4gICAgXG4gICAgcmV0dXJuIG1vZHVsZTtcbn0pOyBcbiJdfQ==
